package easik.states;

import java.util.ArrayList;

import javax.swing.JOptionPane;

import easik.Easik;
import easik.sketch.Sketch;
import easik.sketch.constraint.Constraint;
import easik.sketch.constraint.ProductConstraint;
import easik.sketch.constraint.SumConstraint;
import easik.sketch.path.SketchPath;


/** 
 * This state is used to add a new path to a current constraint
 * 
 * @author Kevin Green 2006
 * @since 2006-08-17 Kevin Green
 * @version 2006-08-23 Kevin Green
 */
public class AddPathToConstraintState extends EasikState implements PathAcceptingState {
	/**
	 * The paths involved in this constraint
	 */
	private ArrayList<SketchPath> _paths;
	/**
	 * The constraint being edited
	 */
	private Constraint _constraint;
	/**
	 * Stores whether the user has finished adding paths to this constraint
	 */
	private boolean _finished;


	/**
	 * Constructor for creating a new state to collect paths and make a diagram.
	 * 
	 * @param inSketch The sketch being worked with.
	 * @param curPaths The current paths of the constraint
	 * @param curConstraint The current constraint
	 */
	public AddPathToConstraintState(Sketch inSketch, ArrayList<SketchPath> curPaths, Constraint curConstraint) {
		_ourSketch = inSketch;
		_finished = false;
		_paths = curPaths;
		_constraint = curConstraint;
	}
	
	/**
	 * Called when a new path has been completed by a state above this on the
	 * stack. If it is null, then cancel, otherwise check to see if the user
	 * selected the 'next' or 'finish' option.
	 * 
	 * @param inPath The last path in, null if it was a cancellation
	 */
	public void passPath(SketchPath inPath) {
		_finished = Easik.getInstance().getStateManager().getFinished();
		//Check if cancelled
		if (inPath == null) {
			Easik.getInstance().getStateManager().popState();
			return;
		}
		
		//Check which path is being passed
		if(!_finished){
			_paths.add(inPath);
			Easik.getInstance().getStateManager().pushState(new GetPathState(true, true));
		}
		
		else{
			_paths.add(inPath);
			if(_constraint instanceof SumConstraint){
				if(SumConstraint.isSumConstraint(_paths)){
					SumConstraint myConst = (SumConstraint)_constraint;
					//Replace previous path array list
					myConst.setPaths(_paths);
					//Remove old tree node
					myConst.getNode().removeFromParent();
					Easik.getInstance().getFrame().getInfoTreeUI().addConstraint(myConst);
					//Referesh Tree
					Easik.getInstance().getFrame().getInfoTreeUI().refreshTree(myConst.getNode());
				}
				else{
					JOptionPane.showMessageDialog(
							Easik.getInstance().getFrame(),
							"Sum constraint could not be changed\n"
								+ "Please ensure that:\n"
								+ "  1) All edges involved target the same entity\n"
								+ "  2) The first edge in each path is injective\n"
								+ "  3) At least two edges are selected\n"
								+ "  4) All paths are unique",
							"Error",
							JOptionPane.ERROR_MESSAGE);
				}
			}
			else if(_constraint instanceof ProductConstraint){
				if(ProductConstraint.isProductConstraint(_paths)){
					ProductConstraint myConst = (ProductConstraint)_constraint;
					//Replace previous path array list
					myConst.setPaths(_paths);
					//Remove old tree node
					myConst.getNode().removeFromParent();
					Easik.getInstance().getFrame().getInfoTreeUI().addConstraint(myConst);
					//Referesh Tree
					Easik.getInstance().getFrame().getInfoTreeUI().refreshTree(myConst.getNode());
				}
				else{
					JOptionPane.showMessageDialog(
							Easik.getInstance().getFrame(),
							"Product constraint could not be created.\n"
								+ "Please ensure that:\n"
								+ "  1) All paths involved have the same source\n"
								+ "  2) At least two paths are selected\n"
								+ "  3) The paths do not form a closed circuit",
							"Error",
							JOptionPane.ERROR_MESSAGE);
				}
			}
			else{
				JOptionPane.showMessageDialog(
						Easik.getInstance().getFrame(),
						"Path addition can not be done on this type of constraint!",
						"Error",
						JOptionPane.ERROR_MESSAGE);
				Easik.getInstance().getStateManager().popState();
			}
			Easik.getInstance().getStateManager().popState();
		}
		
	}

	/**
	 * When this state is pushed on, it sends a message in a popup and then pushes
	 * a path collection state.
	 */
	public void pushedOn() {
		Easik.getInstance().getFrame().getSketch().getSelectionListener().emptySelection();
		Easik.getInstance().getStateManager().resetFinished();
		JOptionPane.showMessageDialog(
			_ourSketch.getParent(),
			"Select a path and then press 'Next' to add it to the constraint and 'Finish' when you are done adding paths",
			"JEase",
			JOptionPane.INFORMATION_MESSAGE);

		Easik.getInstance().getFrame().setStateString("Adding Path(s)");
		setNextButton(true);
		setCancelButton(true);
		setFinishButton(true);
		Easik.getInstance().getStateManager().pushState(new GetPathState(true, true));
	}

	/**
	 * Nothing to do when it gets popped off, as it always pops itself off when completed. 
	 */
	public void poppedOff() {
	}

	/**
	 * This state is called "Adding Path(s)".
	 * 
	 * @return The string literal described above.
	 */
	public String toString() {
		return "Adding Path(s)";
	}

}